<?php

#
# ss_everest_wmic.php
# version 0.4
# June 26, 2009
#
# Copyright (C) 2009, Eric A. Hall
# http://www.eric-a-hall.com/
#
# This software is licensed under the same terms as Cacti itself
#

#
# load the Cacti configuration settings if they aren't already present
#
if (isset($config) == FALSE) {

	if (file_exists(dirname(__FILE__) . "/../include/config.php")) {
		include_once(dirname(__FILE__) . "/../include/config.php");
	}

	if (file_exists(dirname(__FILE__) . "/../include/global.php")) {
		include_once(dirname(__FILE__) . "/../include/global.php");
	}

	if (isset($config) == FALSE) {
		echo ("FATAL: Unable to load Cacti configuration files \n");
		return;
	}
}
#
# call the main function manually if executed outside the Cacti script server
#
if (isset($GLOBALS['called_by_script_server']) == FALSE) {

	array_shift($_SERVER["argv"]);
	print call_user_func_array("ss_everest_wmic", $_SERVER["argv"]);
}

#
# main function
#
function ss_everest_wmic($protocol_bundle="", $sensor_type="",
	$cacti_request="", $data_request="", $data_request_key="") {

	#
	# 1st function argument contains the protocol-specific bundle
	#
	if ((trim($protocol_bundle) == "") || (strpos($protocol_bundle, ":") == FALSE)) {

		echo ("FATAL: No WMI parameter bundle provided\n");
		ss_everest_wmic_syntax();
		return;
	}

	$protocol_array = explode(":", $protocol_bundle);

	if (count($protocol_array) < 3) {

		echo ("FATAL: Not enough elements in WMI parameter bundle\n");
		ss_everest_wmic_syntax();
		return;
	}

	if (count($protocol_array) > 3) {

		echo ("FATAL: Too many elements in WMI parameter bundle\n");
		ss_everest_wmic_syntax();
		return;
	}

	#
	# 1st bundle element is $wmi_hostname
	#
	$wmi_hostname = trim($protocol_array[0]);

	if ($wmi_hostname == "") {

		echo ("FATAL: Hostname not specified in WMI parameter bundle\n");
		ss_everest_wmic_syntax();
		return;
	}

	#
	# 2nd bundle element is $wmi_username
	#
	$wmi_username = $protocol_array[1];

 	if ($wmi_username == "") {

		echo ("FATAL: username must be provided\n");
		ss_everest_wmic_syntax();
		return;
	}

	#
	# 3rd bundle element is $wmi_password (NULL password is okay)
	#
	$wmi_password = $protocol_array[2];

	#
	# 2nd function argument is $sensor_type
	#
	$sensor_type = strtolower(trim($sensor_type));

	if (($sensor_type != "fan") &&
		($sensor_type != "temperature") &&
		($sensor_type != "voltage")) {

		echo ("FATAL: $sensor_type is not a valid sensor type\n");
		ss_everest_wmic_syntax();
		return;
	}

	#
	# 3rd function argument is $cacti_request
	#
	$cacti_request = strtolower(trim($cacti_request));

	if ($cacti_request == "") {

		echo ("FATAL: No Cacti request provided\n");
		ss_everest_wmic_syntax();
		return;
	}

	if (($cacti_request != "index") &&
		($cacti_request != "query") &&
		($cacti_request != "get")) {

		echo ("FATAL: \"$cacti_request\" is not a valid Cacti request\n");
		ss_everest_wmic_syntax();
		return;
	}

	#
	# remaining function arguments are $data_request and $data_request_key
	#
	if (($cacti_request == "query") || ($cacti_request == "get")) {

		$data_request = strtolower(trim($data_request));

		if ($data_request == "") {

			echo ("FATAL: No data requested for Cacti \"$cacti_request\" request\n");
			ss_everest_wmic_syntax();
			return;
		}

		if (($data_request != "sensordevice") &&
			($data_request != "sensorname") &&
			($data_request != "sensorreading")) {

			echo ("FATAL: \"$data_request\" is not a valid data request\n");
			ss_everest_wmic_syntax();
			return;
		}

		#
		# get the index variable
		#
		if ($cacti_request == "get") {

			$data_request_key = strtolower(trim($data_request_key));

			if ($data_request_key == "") {

				echo ("FATAL: No index value provided for \"$data_request\" data request\n");
				ss_everest_wmic_syntax();
				return;
			}
		}

		#
		# clear out spurious command-line parameters on query requests
		#
		else {
			$data_request_key = "";
		}
	}

	#
	# build the wmic command, starting with the location of the wmic executable
	#
	$wmic_command = exec('which wmic 2>/dev/null');

	if ($wmic_command == "") {

		echo ("FATAL: \"wmic\" cannot be found in the user path\n");
		return;
	}

	#
	# append the username
	#
	$wmic_command = $wmic_command . " --user='" . $wmi_username . "'";

	#
	# append the password or set the no-password flag
	#
	if ($wmi_password != "") {

		$wmic_command = $wmic_command . " --password='" . $wmi_password . "'";
	}

	else {
		$wmic_command = $wmic_command . " --no-pass";
	}

	#
	# append the root/wmi namespace where Everest stores values
	#
	$wmic_command = $wmic_command . " --namespace='root/wmi'";

	#
	# append the target hostname
	#
	$wmic_command = $wmic_command . " //" . $wmi_hostname;

	#
	# build the WQL query string based on the Cacti request
	#
	# if they want a listing of sensor devices, ask for the ID column
	#
	if (($cacti_request == "index") || ($data_request == "sensordevice")) {

		$wmic_command = $wmic_command . " 'select ID from Everest_SensorValues";
	}

	#
	# if they want a specific column, ask for it
	#
	else {
		switch ($data_request) {

			#
			# the sensor names
			#
			case "sensorname":

				$wmic_command = $wmic_command . " 'select LABEL from Everest_SensorValues";
				break;

			#
			# the sensor readings
			#
			case "sensorreading":

				$wmic_command = $wmic_command . " 'select VALUE from Everest_SensorValues";
				break;
		}
	}

	#
	# apply a $sensor_type filter on the query
	#
	switch ($sensor_type) {

		case "fan":

			$wmic_command = $wmic_command . " where TYPE = \"F\"";
			break;

		case "temperature":

			$wmic_command = $wmic_command . " where TYPE = \"T\"";
			break;

		case "voltage":

			$wmic_command = $wmic_command . " where TYPE = \"V\"";
			break;
	}

	#
	# if they want a specific sensor, identify it here
	#
	if ($cacti_request == "get") {

		$wmic_command = $wmic_command . " AND ID = \"$data_request_key\"";
	}

	#
	# close the WQL query string
	#
	$wmic_command = $wmic_command . "'";

	#
	# lastly, redirect STDERR to STDOUT so we can trap error text
	#
	$wmic_command = $wmic_command . " 2>&1";

	#
	# run the command
	#
	$wmic_output = exec($wmic_command, $wmic_array);

	#
	# verify that the response contains expected data structures
	#
	if ((isset($wmic_array) == FALSE) ||
		(count($wmic_array) < 3) ||
		(trim($wmic_array[0]) != "CLASS: EVEREST_SensorValues") ||
		(trim($wmic_array[2]) == "")) {

		echo ("FATAL: Incomplete response from wmic");

		#
		# include any response data from wmic if available
		#
		if (trim($wmic_array[0] != "")) {

			echo (" (\"" . substr($wmic_array[0], 0, 32) . "...\")\n");
		}

		elseif (trim($wmic_output) != "") {

			echo (" (\"" . substr($wmic_output, 0, 32) . "...\")\n");
		}

		else {
			echo ("\n");
		}

		return;
	}

	#
	# strip the top two header lines from the wmic response
	#
	array_shift($wmic_array);
	array_shift($wmic_array);

	#
	# create a sensor array from the wmic output
	#
	$sensor_count = 0;

	foreach ($wmic_array as $wmic_response) {

		#
		# if the user requested only the devices, the wmic output will only have the IDs
		#
		if (($cacti_request == "index") || ($data_request == "sensordevice")) {

			$sensor_array[$sensor_count]['index'] = trim($wmic_response);
			$sensor_array[$sensor_count]['name'] = "";
			$sensor_array[$sensor_count]['reading'] = "";
		}

		#
		# otherwise the response data will contain an ID and the requested data, separated by "|"
		#
		else {
			#
			# split the data into device and value columns in $sensor_array
			#
			$scratch = explode("|", $wmic_response, 2);

			$sensor_array[$sensor_count]['index'] = trim($scratch[0]);

			#
			# copy the returned data to the proper location
			#
			if ($data_request == "sensorname") {

				#
				# if the name is unknown, use the device index name
				#
				if (trim($scratch[1]) == "") {

					$scratch[1] = $sensor_array[$sensor_count]['index'];
				}

				#
				# if the name is long and has dashes, trim it down
				#
				while ((strlen($scratch[1]) > 18) && (strrpos($scratch[1], "-") > 12)) {

					$scratch[1] = (substr($scratch[1],0, (strrpos($scratch[1], "-"))));
				}

				#
				# if the name is long and has spaces, trim it down
				#
				while ((strlen($scratch[1]) > 18) && (strrpos($scratch[1], " ") > 12)) {

					$scratch[1] = (substr($scratch[1],0, (strrpos($scratch[1], " "))));
				}

				#
				# if the name is still long, chop it manually
				#
				if (strlen($scratch[1]) > 18) {

					$scratch[1] = (substr($scratch[1],0,17));
				}

				$sensor_array[$sensor_count]['name'] = trim($scratch[1]);;
				$sensor_array[$sensor_count]['reading'] = "";
			}

			if ($data_request == "sensorreading") {

				$sensor_array[$sensor_count]['name'] = "";
				$sensor_array[$sensor_count]['reading'] = trim($scratch[1]);
			}
		}

		#
		# remove malformed readings from the current row's value field in $sensor_arrayso
		#
		# the readings must be removed instead of zeroed, so that RRD will store a NULL
		#
		if ($data_request == "sensorreading") {

			#
			# remove non-numeric sensor readings
			#
			if (!is_numeric($sensor_array[$sensor_count]['reading'])) {

				$sensor_array[$sensor_count]['reading'] = "";
			}

			#
			# remove impossibly-high temperature and voltage readings
			#
			if ((($sensor_type == "temperature") || ($sensor_type == "voltage")) &&
				($sensor_array[$sensor_count]['reading'] >= "255")) {

				$sensor_array[$sensor_count]['reading'] = "";
			}
		}

		#
		# increment the sensor counter
		#
		$sensor_count++;
	}

	#
	# verify that the sensor_array exists and has data
	#
	if ((isset($sensor_array) == FALSE) ||
		(count($sensor_array) == 0)) {

		echo ("FATAL: No matching sensors were returned from SNMP\n");
		return;
	}

	#
	# generate output
	#
	foreach ($sensor_array as $sensor) {

		#
		# return output data according to $cacti_request
		#
		switch ($cacti_request) {

			#
			# for "index" requests, dump the device column
			#
			case "index":

				echo ($sensor['index'] . "\n");
				break;

			#
			# for "query" requests, dump the requested columns
			#
			case "query":

				switch ($data_request) {

					case "sensordevice":

						echo ($sensor['index'] . ":" . $sensor['index'] . "\n");
						break;

					case "sensorname":

						echo ($sensor['index'] . ":" . $sensor['name'] . "\n");
						break;

					case "sensorreading":

						echo ($sensor['index'] . ":" . $sensor['reading'] . "\n");
						break;
				}

				break;

			#
			# for "get" requests, dump the requested data for the requested sensor
			#
			case "get":

				#
				# skip the current row if it isn't the requested sensor
				#
				if (strtolower($sensor['index']) != $data_request_key) {

					break;
				}

				switch ($data_request) {

					case "sensordevice":

						echo ($sensor['index'] . "\n");
						break;

					case "sensorname":

						echo ($sensor['name'] . "\n");
						break;

					case "sensorreading":

						if (isset($GLOBALS['called_by_script_server']) == TRUE) {

							return($sensor['reading']);
						}

						else {
							echo ($sensor['reading'] . "\n");
						}

						break;
				}

				break;
		}
	}
}

#
# display the syntax
#
function ss_everest_wmic_syntax() {

	echo ("Usage: ss_everest_wmic.php <hostname>:<wmi_username>:[<wmi_password>] \ \n" .
	"      (FAN|TEMPERATURE|VOLTAGE) (index|query <fieldname>|get <fieldname> <sensor>) \n");
}

?>
