<?php

#
# ss_wbem_sensors.php
# version 0.1
# November 11, 2010
#
# Copyright (C) 2010, Eric A. Hall
# http://www.eric-a-hall.com/
#
# This software is licensed under the same terms as Cacti itself
#

#
# load the Cacti configuration settings if they aren't already present
#
if (isset($config) == FALSE) {

	if (file_exists(dirname(__FILE__) . "/../include/config.php")) {
		include_once(dirname(__FILE__) . "/../include/config.php");
	}

	if (file_exists(dirname(__FILE__) . "/../include/global.php")) {
		include_once(dirname(__FILE__) . "/../include/global.php");
	}

	if (isset($config) == FALSE) {
		echo ("FATAL: Unable to load Cacti configuration files \n");
		return;
	}
}
#
# call the main function manually if executed outside the Cacti script server
#
if (isset($GLOBALS['called_by_script_server']) == FALSE) {

	array_shift($_SERVER["argv"]);
	print call_user_func_array("ss_wbem_sensors", $_SERVER["argv"]);
}

#
# main function
#
function ss_wbem_sensors($protocol_bundle="", $sensor_type="",
	$cacti_request="", $data_request="", $data_request_key="") {

	#
	# 1st function argument contains the protocol-specific bundle
	#
	if ((trim($protocol_bundle) == "") || (strpos($protocol_bundle, ":") == FALSE)) {

		echo ("FATAL: No WBEM parameter bundle provided\n");
		ss_wbem_sensors_syntax();
		return;
	}

	$protocol_array = explode(":", $protocol_bundle);

	if (count($protocol_array) < 5) {

		echo ("FATAL: Not enough elements in WBEM parameter bundle\n");
		ss_wbem_sensors_syntax();
		return;
	}

	if (count($protocol_array) > 5) {

		echo ("FATAL: Too many elements in WBEM parameter bundle\n");
		ss_wbem_sensors_syntax();
		return;
	}

	#
	# 1st bundle element is $wbem_hostname
	#
	$wbem_hostname = trim($protocol_array[0]);

	if ($wbem_hostname == "") {

		echo ("FATAL: Hostname not specified in WBEM parameter bundle\n");
		ss_wbem_sensors_syntax();
		return;
	}

	#
	# 2nd bundle element is $wbem_transport
	#
	$wbem_transport = strtolower($protocol_array[1]);

 	if ($wbem_transport == "") {

		#
		# default is HTTPS
		#
		$wbem_transport = "https";
	}

	if (($wbem_transport != "http") && ($wbem_transport != "https")) {

		echo ("FATAL: Unknown transport \"$wbem_transport\" specified in WBEM bundle\n");
		ss_wbem_sensors_syntax();
		return;
	}

	#
	# 3rd bundle element is $wbem_port
	#
	$wbem_port = trim($protocol_array[2]);

	if ($wbem_port == "") {

		#
		# use the default port for the selected transport
		#
		switch ($wbem_transport) {

			case "http":
				$wbem_port = 5988;
				break;

			case "https":
				$wbem_port = 5989;
				break;
		}
	}

	if (is_numeric($wbem_port) == FALSE) {

		echo ("FATAL: Non-numeric port \"$wbem_port\" specified in WBEM bundle\n");
		ss_wbem_sensors_syntax();
		return;
	}

	#
	# 2nd bundle element is $wbem_username
	#
	$wbem_username = $protocol_array[3];

 	if ($wbem_username == "") {

		echo ("FATAL: username must be provided\n");
		ss_wbem_sensors_syntax();
		return;
	}

	#
	# 3rd bundle element is $wbem_password (NULL password is okay)
	#
	$wbem_password = $protocol_array[4];

	#
	# 2nd function argument is $sensor_type
	#
	$sensor_type = strtolower(trim($sensor_type));

	if (($sensor_type != "fan") &&
		($sensor_type != "temperature") &&
		($sensor_type != "voltage")) {

		echo ("FATAL: $sensor_type is not a valid sensor type\n");
		ss_wbem_sensors_syntax();
		return;
	}

	#
	# 3rd function argument is $cacti_request
	#
	$cacti_request = strtolower(trim($cacti_request));

	if ($cacti_request == "") {

		echo ("FATAL: No Cacti request provided\n");
		ss_wbem_sensors_syntax();
		return;
	}

	if (($cacti_request != "index") &&
		($cacti_request != "query") &&
		($cacti_request != "get")) {

		echo ("FATAL: \"$cacti_request\" is not a valid Cacti request\n");
		ss_wbem_sensors_syntax();
		return;
	}

	#
	# remaining function arguments are $data_request and $data_request_key
	#
	if (($cacti_request == "query") || ($cacti_request == "get")) {

		$data_request = strtolower(trim($data_request));

		if ($data_request == "") {

			echo ("FATAL: No data requested for Cacti \"$cacti_request\" request\n");
			ss_wbem_sensors_syntax();
			return;
		}

		if (($data_request != "sensordevice") &&
			($data_request != "sensorname") &&
			($data_request != "sensorreading")) {

			echo ("FATAL: \"$data_request\" is not a valid data request\n");
			ss_wbem_sensors_syntax();
			return;
		}

		#
		# get the index variable
		#
		if ($cacti_request == "get") {

			$data_request_key = strtolower(trim($data_request_key));

			if ($data_request_key == "") {

				echo ("FATAL: No index value provided for \"$data_request\" data request\n");
				ss_wbem_sensors_syntax();
				return;
			}
		}

		#
		# clear out spurious command-line parameters on query requests
		#
		else {
			$data_request_key = "";
		}
	}

	#
	# determine if the wbemcli executable is in the path
	#
	$wbemcli_command = exec('which wbemcli 2>/dev/null');

	if ($wbemcli_command == "") {

		echo ("FATAL: \"wbemcli\" cannot be found in the user path\n");
		return;
	}

	#
	# we don't know the full command yet, but we know the options so start with those
	#
	# use newlines for each property pair, and do not verify the SSL certificate
	#
	$wbemcli_command = " -nl -noverify";

	#
	# construct the URL
	#
	$wbemcli_command = $wbemcli_command . " '" . $wbem_transport . "://";

	#
	# append the username
	#
	$wbemcli_command = $wbemcli_command . $wbem_username;

	#
	# append the password
	#
	if ($wbem_password != "") {

		$wbemcli_command = $wbemcli_command . ":" . $wbem_password;
	}

	else {
		$wbemcli_command = $wbemcli_command . ":" . "\"\"";
	}

	#
	# append the target hostname
	#
	$wbemcli_command = $wbemcli_command . "@" . $wbem_hostname;

	#
	# append the target port number
	#
	$wbemcli_command = $wbemcli_command . ":" . $wbem_port;

	#
	# append the full namespace for the OMC_NumericSensor instances
	#
	$wbemcli_command = $wbemcli_command . "/root/cimv2:OMC_NumericSensor";

	#
	# if they want all of the sensor devices, use ei to enumerate the instances, and
	# close the URL
	#
	if (($cacti_request == "index") || ($cacti_request == "query")) {

		$wbemcli_command = exec('which wbemcli 2>/dev/null') . " ei" .
			$wbemcli_command . "'";
	}

	#
	# they want a specific property for a specific device, so use gi to request the
	# instance, and then close the URL
	#
	else {
		$wbemcli_command = exec('which wbemcli 2>/dev/null') . " gi" .
			$wbemcli_command . ".DeviceID=\"" . $data_request_key . "\"" . "'";
	}

	#
	# tell wbemcli to only return the interesting properties
	#
	$wbemcli_command = $wbemcli_command . " ElementName,SensorType,CurrentReading";

	#
	# lastly, redirect STDERR to STDOUT so we can trap error text
	#
	$wbemcli_command = $wbemcli_command . " 2>&1";

	#
	# run the command
	#
	$wbemcli_output = exec($wbemcli_command, $wbemcli_array);

	#
	# verify that the response contains expected data structures
	#
	if ((isset($wbemcli_array) == FALSE) ||
		(substr($wbemcli_array[0], 0, strlen($wbem_hostname)) != $wbem_hostname)) {

		#
		# include any response data from wbemcli if available
		#
		if (isset($wbemcli_array[1])) {

			echo ("FATAL: " . (substr($wbemcli_array[1], 
				(strpos($wbemcli_array[1], "wbemcli:") + 9),
				(strpos($wbemcli_array[1], "wbemcli:") + 40))) . "\n");
		}

		else {
			echo ("FATAL: No response from wbemcli");
		}

		return;
	}

	#
	# create a sensor array from the wbemcli output
	#
	$sensor_count = 0;

	foreach ($wbemcli_array as $wbemcli_response) {

		#
		# process the DeviceID
		#
		if (preg_match("/DeviceID=\"(\S+)\"$/", $wbemcli_response, $scratch) == 1) {

			$sensor_array[$sensor_count]['index'] = $scratch[1];
		}

		#
		# process the sensor name
		#
		elseif (preg_match("/^-ElementName=(.*)$/", $wbemcli_response, $scratch) == 1) {

			#
			# strip surrounding double-quote marks from the name
			#
			if (substr($scratch[1], 0, 1) == "\"") {

				$scratch[1] = substr($scratch[1], 1);
			}

			if (substr($scratch[1], (strlen($scratch[1]) - 1), 1) == "\"") {

				$scratch[1] = substr($scratch[1], 0, (strlen($scratch[1]) - 1));
			}

			$sensor_array[$sensor_count]['name'] = $scratch[1];
		}

		#
		# process the sensor type
		#
		elseif (preg_match("/^-SensorType=(\d)$/", $wbemcli_response, $scratch) == 1) {

			switch ($scratch[1]) {

				case 2:
					$sensor_array[$sensor_count]['type'] = "temperature";
					break;
			
				case 3:
					$sensor_array[$sensor_count]['type'] = "voltage";
					break;
			
				case 5:
					$sensor_array[$sensor_count]['type'] = "fan";
					break;

				default:
					$sensor_array[$sensor_count]['type'] = "unknown";
					break;
			}
		}

		#
		# process the sensor reading
		#
		elseif (preg_match("/^-CurrentReading=(.*)$/", $wbemcli_response, $scratch) == 1) {

			$sensor_array[$sensor_count]['reading'] = $scratch[1];
		}

		#
		# an empty row marks the end of sensor data
		#
		# verify and purge data as needed
		#
		elseif ($wbemcli_response == "") {

			#
			# remove non-numeric sensor readings to make RRD record a NULL
			#
			if (!is_numeric($sensor_array[$sensor_count]['reading'])) {

				$sensor_array[$sensor_count]['reading'] = "";
			}

			#
			# normalize numeric readings by type
			#
			else {
				#
				# shift the decimal place by two
				#
				$sensor_array[$sensor_count]['reading'] =
					($sensor_array[$sensor_count]['reading'] * .01);

				#
				# remove impossibly-high temperature and voltage readings
				#
				if ((($sensor_array[$sensor_count]['type'] == "temperature") ||
					($sensor_array[$sensor_count]['type'] == "voltage")) &&
					($sensor_array[$sensor_count]['reading'] >= "255")) {

					$sensor_array[$sensor_count]['reading'] = "";
				}
			}

			#
			# discard entries of the wrong type, and reuse sensor_count
			#
			if ($sensor_array[$sensor_count]['type'] != $sensor_type) {

				array_pop($sensor_array);
			}

			#
			# otherwise increment the sensor counter
			#
			else {
				$sensor_count++;
			}
		}

		#
		# abort if a preg_match() failed or if we encounter unplanned data
		#
		else {
			echo ("FATAL: Unexpected response data from wbemcli\n");
			return;
		}
	}

	#
	# verify that the sensor_array exists and has data
	#
	if ((isset($sensor_array) == FALSE) ||
		(count($sensor_array) == 0)) {

		echo ("FATAL: No matching sensors were returned from wbemcli\n");
		return;
	}

	#
	# generate output
	#
	foreach ($sensor_array as $sensor) {

		#
		# return output data according to $cacti_request
		#
		switch ($cacti_request) {

			#
			# for "index" requests, dump the device column
			#
			case "index":

				echo ($sensor['index'] . "\n");
				break;

			#
			# for "query" requests, dump the requested columns
			#
			case "query":

				switch ($data_request) {

					case "sensordevice":

						echo ($sensor['index'] . ":" . $sensor['index'] . "\n");
						break;

					case "sensorname":

						echo ($sensor['index'] . ":" . $sensor['name'] . "\n");
						break;

					case "sensorreading":

						echo ($sensor['index'] . ":" . $sensor['reading'] . "\n");
						break;
				}

				break;

			#
			# for "get" requests, dump the requested data for the requested sensor
			#
			case "get":

				#
				# skip the current row if it isn't the requested sensor
				#
				if (strtolower($sensor['index']) != $data_request_key) {

					break;
				}

				switch ($data_request) {

					case "sensordevice":

						echo ($sensor['index'] . "\n");
						break;

					case "sensorname":

						echo ($sensor['name'] . "\n");
						break;

					case "sensorreading":

						if (isset($GLOBALS['called_by_script_server']) == TRUE) {

							return($sensor['reading']);
						}

						else {
							echo ($sensor['reading'] . "\n");
						}

						break;
				}

				break;
		}
	}
}

#
# display the syntax
#
function ss_wbem_sensors_syntax() {

	echo ("Usage: ss_wbem_sensors.php <hostname>:<transport>:<port>:<username>:[<password>] \ \n" .
	"      (FAN|TEMPERATURE|VOLTAGE) (index|query <fieldname>|get <fieldname> <sensor>) \n");
}

?>
